/**
*
* \file        versiport.c
* \brief       versiport driver (hardware independent functions
* \author      Larry Salant
* \date        5/31/2011
*/

////////////////////////////////////////////////////////////////////////////////
#include "nutypedefs.h"
#include "cresnet.h"
#include "string_utils.h"
#include <string.h>
#include <stdlib.h>
#include "console.h"
#include "os.h"
#include "cresnet_slave.h"
#include "errors.h"
#include "product.h"
#include "DeviceJoinInterface.h"
#include "versiport.h"

void VersiportWrapperTask(UINT32 pVersiport);

////////////////////////////////////////////////////////////////////////////////
/**
* \author      Larry Salant
* \date        5/31/2011
* \brief       Constructor for versiport driver
* \param       number of versiports on the board
* \return      void
* \retval      none
*/
CVersiportDriver::CVersiportDriver(UINT8 NumPorts, UINT8 ThreadPriority, UINT8 Stream)
{
    int port;
	VERSIPORT_DATA *pPort;

	m_NumPorts = NumPorts;
	m_Stream = Stream;
	m_ReportChanges = false;
#ifdef DEBUG        
	mDebug = 0;
#endif // DEBUG
        
	// allocate memory for port data
	m_pVdata = (VERSIPORT_DATA *)malloc(NumPorts*sizeof(VERSIPORT_DATA));
	pPort = m_pVdata;

	// set all parameters to default values
    for ( port = 0; port < m_NumPorts; port++)
    {
        pPort->m_iInputConfig = VERSIPORT_NONE;
        pPort->m_iAnalogInput = 0;
        pPort->m_iAnalogHysteresis = DEFAULT_HYSTERESIS;
        pPort->m_iLastReported = 0;
        pPort->m_bForceReport = false;
        ++pPort;
    }
    
	// set the hardware to the default state
	InitHardware();

    //create critical section
    m_pVportCS = new CriticalSection();  // to protect the device list
    // create task to monitor ADC's
    OsCreateNamedAdvTask(VersiportWrapperTask, OsGetDefaultStackVarCnt(), (UINT32*)this, ThreadPriority, "VPORT");
}

/**
* \author    Larry Salant
* \brief     set hysteresis for ADC (min change)
* \date      3/28/2011
* \param     Verisport join
* \param     value
* \return    void
**/
UINT8 CVersiportDriver::SetHysteresis(UINT32 Join, UINT32 value)
{
    UINT8 valid = 0;

// The join offset for series 3 versiports is 64 instead of 32
#ifndef SERIES_3_CONTROL
    // if the command is to set the state
    Join -= HYSTERESIS_ANLG_0;
#else
    // Check join first before modifying it to fit the right port
    if (Join >= SERIES_3_MIN_CHANGE_OFFSET)
      Join -= SERIES_3_MIN_CHANGE_OFFSET;
    else
      return valid;
#endif
    
    if (Join < m_NumPorts)
    {
        // set the hysteresis value
        (m_pVdata+Join)->m_iAnalogHysteresis = value;
        valid = 1;
    }
   return valid;
}

/**
* \author    Larry Salant
* \brief     Periodic task to monitor versiports
* \date      6/1/2011
* \param     none
* \return    void
**/
void CVersiportDriver::VersiportUpdateTask(void)
{
    UINT8 port;
	CresnetJoinState *pGetState =  GetState(m_Stream);

    // not called from the timer task since it may block on the critical section
    while (1)
    {
        // sleep for a while
        HwDelayMsec(VERSIPORT_POLL_TIME);

        // read all the ADC values
        ReadAdcs();

        // if the program is not running, don't report any changes
#ifdef DEBUG        
        if (m_ReportChanges || mDebug) 
#else
        if (m_ReportChanges) 
#endif // DEBUG
        {
            UINT16 diff;
			VERSIPORT_DATA *pPort = m_pVdata;

            // enter critical section
            m_pVportCS->Enter();

            for (port = 0; port < m_NumPorts; port++) 
            {
                bool changed = pPort->m_bForceReport; // Force a report if needed
                pPort->m_bForceReport = false; // Reset bool

                switch (pPort->m_iInputConfig) 
                {
                    // port configured as analog input
                case VERSIPORT_ANLG_IN: 
                    // If not forcing a report, check if there are any changes in the value
                    if (!changed)
                    {
                      // m_iLastReported  was the ADC value when we last reported it, see if the value changed
                      if (pPort->m_iAnalogInput >= pPort->m_iLastReported)
                        diff = pPort->m_iAnalogInput - pPort->m_iLastReported;
                      else
                        diff = pPort->m_iLastReported - pPort->m_iAnalogInput;
                         
                      // if the value changed by more than the hysteresis OR it was clipped on either rail,
                      if ((diff > pPort->m_iAnalogHysteresis) ||
                          ((diff > 0) && ((pPort->m_iAnalogInput == 0) || (pPort->m_iAnalogInput == 0xffff))))
                      {  
                        changed = true;
                      }
                    }

                    // if the value has changed
                    if (changed)
                    {
                        // report it.
#ifdef DEBUG
                        if (mDebug)
                            DmConsolePrintf("VERSIPORT %d: was %u  is %u\n", port, pPort->m_iLastReported, pPort->m_iAnalogInput);                         
                        else
                            pGetState->CresnetAnalogMemorize(VERSIPORT_INPUT_OFFSET+port, pPort->m_iAnalogInput, GetNetDestination(), TRUE);
#else
                        // Force update since the joins won't be sent out if they did not change (same program resets)
                        pGetState->CresnetAnalogMemorize(VERSIPORT_INPUT_OFFSET+port, pPort->m_iAnalogInput, GetNetDestination(), TRUE);
#endif // DEBUG
                        // remember new value
                        pPort->m_iLastReported = pPort->m_iAnalogInput;
                    }
                    break;

                case VERSIPORT_DIGI_IN :		// configured as digital input
                    {
                        UINT16 CurrentValue = pPort->m_iLastReported;

                        // determine if the input is a 1 or 0
                        if ( pPort->m_iAnalogInput < DIGI_LO_TRESH ) 
                            CurrentValue = 1;
                        else if ( pPort->m_iAnalogInput > DIGI_HI_TRESH )
                            CurrentValue = 0;
                        // if we haven't reported it yet, or it's changed,
                        if (changed || (pPort->m_iLastReported != CurrentValue))
                        {
                          // Force update since the joins won't be sent out if they did not change (same program resets)
                          pGetState->CresnetDigitalMemorize(VERSIPORT_INPUT_OFFSET+port, CurrentValue, GetNetDestination(), TRUE);
                          pPort->m_iLastReported = CurrentValue;
                        }
                    }
                    break;
                case VERSIPORT_NONE:
                case VERSIPORT_DIGI_OUT:
                default:
                    break;
                } // switch (pPort->m_iInputConfig) 
				++pPort;
            } //for (port = 0; port < m_NumPorts; port++) 
            // Leave critical section
            m_pVportCS->Leave();
        } // if (m_ReportChanges)
    }
}

/**
* \author  Larry Salant
* \brief   C wrapper for Main task for versiport
* \date    6/1/2011
* \param   pointer to versiport driver
* \retval  void
*/
void VersiportWrapperTask(UINT32 pVersiport)
{
    // go to the main loop; will not return until driver is shutdown
    ((CVersiportDriver*)pVersiport)->VersiportUpdateTask();

    // Exit Thread
}

#ifdef DEBUG
/**
* \author      Larry Salant
* \date        5/31/2011
* \brief       Processes Configuration packet for Versiport
* \param       pPacket : pointer to the packet
* \return      void
* \retval      none
*/
void CVersiportDriver::VersiportDebug(UINT8 test)
{
    mDebug = test;
    int port;

    if (mDebug == 1)  // ADC test
    {
        // set all parameters to default values
        for ( port = 0; port < m_NumPorts; port++)
        {
            (m_pVdata+port)->m_iInputConfig = VERSIPORT_ANLG_IN;
        }
    }
	else if (mDebug == 2)
	{
        int port;

        // set all parameters to default values
        for ( port = 0; port < m_NumPorts; port++)
        {
            (m_pVdata+port)->m_iInputConfig = VERSIPORT_ANLG_IN;
        }
		ReadAdcs();

		// report the current adc values
        for ( port = 0; port < m_NumPorts; port++)
        {
            DmConsolePrintf("VERSIPORT adc %d:  is %u\n", port, (m_pVdata+port)->m_iAnalogInput);                         
        }
		mDebug = 0;
	}
    else if (mDebug == 3)  // set all to di
    {
        // set all parameters to default values
        for ( port = 0; port < m_NumPorts; port++)
        {
            (m_pVdata+port)->m_iInputConfig = VERSIPORT_DIGI_IN;
        }
    }
     else if (mDebug == 4)  // set all to do
    {
        // set all parameters to default values
        for ( port = 0; port < m_NumPorts; port++)
        {
            (m_pVdata+port)->m_iInputConfig = VERSIPORT_DIGI_OUT;
        }
    }
   
}
#endif // DEBUG

